<?php
require_once("lib/stripe-php/init.php"); // Stripe SDK Import

/*
 * This action hook registers our PHP class as a WooCommerce payment gateway
 */
add_filter( 'woocommerce_payment_gateways', 'quivers_add_gateway_class' );
function quivers_add_gateway_class( $gateways ) {
	$gateways[] = 'WC_Quivers_Gateway'; // your class name is here
	return $gateways;
}

/*
 * The class itself, please note that it is inside plugins_loaded action hook
 */

// Checked Woocommerce plugin is enabled or not
include_once( ABSPATH . 'wp-admin/includes/plugin.php' );

if ( is_plugin_active( 'woocommerce/woocommerce.php' ) ){

	// Woocommerce Plugin is enabled
add_action( 'plugins_loaded', 'quivers_init_gateway_class' );

function quivers_init_gateway_class() {
	class WC_Quivers_Gateway extends WC_Payment_Gateway {
		/**
 		 * Class constructor
 		 */
 		public function __construct() {
 
			$this->id = 'quivers_stripe'; // payment gateway plugin ID
			$this->icon = ''; // URL of the icon that will be displayed on checkout page near your gateway name
			$this->has_fields = true; // in case you need a custom credit card form
			$this->method_title = 'Quivers-Pay Stripe';
			$this->method_description = 'Description of Quivers payment gateway'; // will be displayed on the options page

			$this->supports = array(
				'products'
			);
		 
			// Method with all the options fields
			$this->init_form_fields();
		 
			// Load the settings.
			$this->init_settings();
			$this->title = $this->get_option( 'title' );
			$this->description = $this->get_option( 'description' );
			$this->enabled = $this->get_option( 'enabled' );
			$this->testmode = 'yes' === $this->get_option( 'testmode' );
			$this->private_key = $this->get_option( 'private_key' );
			$this->publishable_key = $this->get_option( 'publishable_key' );
			$this->min_order_total = $this->get_option( 'min_order_total' );
		
			
			
			// This action hook saves the settings
			add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
		 
			// We need custom JavaScript to obtain a token
			add_action( 'wp_enqueue_scripts', array( $this, 'payment_scripts' ) ); 
 		}
 		/**
 		 * Checks if gateway should be available to use.
 		 */
 		public function is_available() {
 			if ( WC()->cart && $this->get_order_total() < $this->min_order_total ) {
 				return false;
 			}

 			return parent::is_available();
		 }

 
		/**
 		 * Plugin options
 		 */
 		public function init_form_fields(){
			
			$this->form_fields = array(
				'enabled' => array(
					'title'       => 'Enable/Disable',
					'label'       => 'Enable Quivers Gateway',
					'type'        => 'checkbox',
					'description' => '',
					'default'     => 'no'
				),
				'title' => array(
					'title'       => 'Title',
					'type'        => 'text',
					'description' => 'This controls the title which the user sees during checkout.',
					'default'     => 'Pay by Card',
					'desc_tip'    => true,
				),
				'description' => array(
					'title'       => 'Description',
					'type'        => 'textarea',
					'description' => 'This controls the description which the user sees during checkout.',
					'default'     => 'Pay with your credit card via our super-cool payment gateway.',
				),
				'testmode' => array(
					'title'       => 'Test mode',
					'label'       => 'Enable Test Mode',
					'type'        => 'checkbox',
					'description' => 'Place the payment gateway in test mode using test API keys.',
					'default'     => 'yes',
					'desc_tip'    => true,
				),
				'publishable_key' => array(
					'title'       => 'Live Publishable Key',
					'type'        => 'text'
				),
				'private_key' => array(
					'title'       => 'Live Private Key',
					'type'        => 'password'
				),
				'min_order_total' => array(
					'title'       => 'Minimum Order Total',
					'type'        => 'text',
					'description' => '$0.50 is the minimum amount allowed by Stripe',
					'default'     => '0.50'
				),
			);
 
	 	}
 
		/**
		 * You will need it if you want your custom credit card form
		 */
		public function payment_fields() {
				WC()->session->set('stripe_enable',$this->enabled);
				WC()->session->set('private_key',$this->private_key);
			
 			// ok, let's display some description before the payment form
			if ( $this->description ) {
				// you can instructions for test mode, I mean test card numbers etc.
				if ( $this->testmode ) {
					$this->description .= ' TEST MODE ENABLED. In test mode, you can use the card numbers listed in <a href="#" target="_blank">documentation</a>.';
					$this->description  = trim( $this->description );
				}
				// display the description with <p> tags etc.
				echo wpautop( wp_kses_post( $this->description ) );
			}
		 	
		 	// I will echo() the form, but you can close PHP tags and print it directly in HTML
	 		echo '<fieldset id="wc-' . esc_attr( $this->id ) . '-cc-form" class="wc-credit-card-form wc-payment-form" style="background:transparent;">';
	 	 
	 		// Add this action hook if you want your custom gateway to support it
	 		do_action( 'woocommerce_credit_card_form_start', $this->id );
	 	 
	 		// I recommend to use inique IDs, because other gateways could already use #ccNo, #expdate, #cvc
	 		$html = <<<HTML

	 		<style type="text/css">
	 		/**
	 		* The CSS shown here will not be introduced in the Quickstart guide, but
	 		* shows how you can use CSS to style your Element's container.
	 		*/
	 		input,
	 		.StripeElement {
	 		  height: 40px;
	 		  padding: 10px 12px;

	 		  color: #32325d;
	 		  background-color: white;
	 		  border: 1px solid transparent;
	 		  border-radius: 4px;

	 		  box-shadow: 0 1px 3px 0 #e6ebf1;
	 		  -webkit-transition: box-shadow 150ms ease;
	 		  transition: box-shadow 150ms ease;
	 		}
			 .grecaptcha-badge {display: none;}  
	 		input:focus,
	 		.StripeElement--focus {
	 		  box-shadow: 0 1px 3px 0 #cfd7df;
	 		}

	 		.StripeElement--invalid {
	 		  border-color: #fa755a;
	 		}

	 		.StripeElement--webkit-autofill {
	 		  background-color: #fefde5 !important;
	 		}
	 		</style>
			
 			<div class="form-row">
				<label for="card-element">
			      Credit or debit card
			    </label>
			<!-- <div id="card-element"> (changes done for qt-3799) -->
			<div id="card-element" style="width: 100%">
				<script>
					init_stripe();
				</script>
				<!-- A Stripe Element will be inserted here. -->
			</div>
			
			<!-- Used to display form errors. -->
			<div id="card-errors" role="alert"></div>
			</div>
			<input type="hidden" id="stripe_payment_token" value="" />
HTML;
			echo $html;
	 		do_action( 'woocommerce_credit_card_form_end', $this->id );
	 	 
	 		echo '<div class="clear"></div></fieldset>';
		}
 
		/*
		 * Custom CSS and JS, in most cases required only when you decided to go with a custom credit card form
		 */
	 	public function payment_scripts() {	
 			// we need JavaScript to process a token only on cart/checkout pages, right?
			if ( ! is_cart() && ! is_checkout() && ! isset( $_GET['pay_for_order'] ) ) {
				return;
			}
		 
			// if our payment gateway is disabled, we do not have to enqueue JS too
			if ( 'no' === $this->enabled ) {
				WC()->session->set('stripe_enable',$this->enabled);
				return;
			}
		 
			// no reason to enqueue JavaScript if API keys are not set
			if ( empty( $this->private_key ) || empty( $this->publishable_key ) ) {
				return;
			}
		 
			// do not work with card detailes without SSL unless your website is in a test mode
			if ( ! $this->testmode && ! is_ssl() ) {
				return;
			}

			// let's suppose it is our payment processor JavaScript that allows to obtain a token
			wp_enqueue_script( 'stripe_js', 'https://js.stripe.com/v3/' );
 
			// and this is our custom JS in your plugin directory that works with token.js
			wp_register_script( 'quivers_stripe_js', plugins_url( 'quivers_stripe.js', __FILE__ ), array( 'jquery', 'stripe_js' ) );
		 
			// in most payment processors you have to use PUBLIC KEY to obtain a token
			wp_localize_script( 'quivers_stripe_js', 'quivers_stripe_params', array(
				'publishableKey' => $this->publishable_key
			) );
		 
			wp_enqueue_script( 'quivers_stripe_js' );
			// if(isset($_REQUEST['type_error']) && $_REQUEST['type_error']=='invalid_request_error') {
			// 	WC()->session->set('clkey',null);
			//     WC()->session->set('intent_id',null);
			// 	wc_add_notice( 'We are unable to Authenticate your payment method. Looks like the selected payment method has not been configured correctly or Please choose a different payment method and try again. If the issue persists please contact Quivers at 805-250-4555.', 'error' );
				
			// 	return array(
			// 		'result'   => 'fail',
			// 		'redirect' => '',
			// 		'refresh' => true
			// 	);
			// 	echo "<script>$('body').scrollTop();</script>";
			// } 
	 	}
 
		/*
 		 * Fields validation
		 */
		public function validate_fields() {
			return true;
		}
 
		/*
		 * We're processing the payments here
		 */
		public function process_payment( $order_id ) {
			$log = new My_WC_LogTracking();
			$quivers_debug_mode = get_option('wc_settings_tab_debug_key');
			try {
				$order = wc_get_order( $order_id );
				$statement_descriptor = WC()->session->get('statement_descriptor_suffix');
				$destination = WC()->session->get('destination');
                $merchantId = WC()->session->get('merchantId');
			}
			catch(Exception $e) {
				wc_add_notice( $e->getLocalizedMessage(), 'error' );
				/* translators: error message */
				$order->update_status( 'failed' );

				return array(
					'result'   => 'fail',
					'redirect' => '',
				);
			}

			// Set API key
			\Stripe\Stripe::setApiKey($this->private_key);

			if(WC()->session->get('total_amount')){
				$amount = WC()->session->get('total_amount');
			}
			else{
				$amount = $order->get_total();
			}
			$validate_response = WC()->session->get('response');
			$currency = $order->get_currency();
			$chargeAmount = $this->formatCurrencyAmount($amount, $currency);
        	$customerEmail = $order->get_billing_email();
			$email = WC()->customer->get_billing_email();
			if($quivers_debug_mode){
				$log->debug_mode("process_payment", "Amount", $amount, $email,null, null);
			}
        	if ( ($customerEmail == NULL) || ($customerEmail == 'undefined') )
        	{
        		// Get Email from Customer
        		$customer_id = get_current_user_id();
				if ($customer_id != 0)
        		{
        			$customerEmail = get_userdata($customer_id)->user_email;
        		}
        	}

        	// If Email is not retrieved, then show an error. 
        	if ( ($customerEmail == NULL) || ($customerEmail == 'undefined') )
        	{
        		wc_add_notice( "Billing Email is required!", 'error' );
        		/* translators: error message */
        		$order->update_status( 'failed' );
        		return array(
        			'result'   => 'fail',
        			'redirect' => '',
        			'refresh' => true
        		);
        	}

			$shippingInfo = array(
				'address' => array(
					'line1' => $order->get_shipping_address_1(),
					'line2' => $order->get_shipping_address_2(),
					'city' => $order->get_shipping_city(),
					'state' => $order->get_shipping_state(),
					'country' => $order->get_shipping_country(),
					'postal_code' => $order->get_shipping_postcode(),
				),
				'name' => sprintf("%s %s",$order->get_shipping_first_name(), $order->get_shipping_last_name())
			);

        	try {
				// get or create Stripe customer
				$customer = false;
	            $customersList = \Stripe\Customer::all(array(
	                "limit" => 1,
	                "email" => $customerEmail,
	            ));

	            if(count($customersList["data"]) > 0) {
	                $customer = $customersList["data"][0];
	            }
				try{
					$payment_source = $_POST['stripe_source'];
				}
				catch(Exception $e) {
					wc_add_notice( "Unable to recieve payment details.", 'error' );
					/* translators: error message */
					return array(
						'result'   => 'fail',
						'redirect' => '',
						'refresh' => true
					);
				}
	            if (!$customer) {
					$customer = \Stripe\Customer::create(array(
		                'source' => $payment_source,
		                'email' => $customerEmail,
		                'shipping' => $shippingInfo,
	                ));
	            }

	            // add source to customer
	            $customerId = $customer["id"];
                try {
                    $customer->sources->create(array("source" => $payment_source));
                }
                catch (Exception $e) {}
	            
				// create charge
		

			/* Intent Request*/
            global $woocommerce;
			$amount = WC()->session->get('total_amount');
            $amount = $amount  == null || $amount == 0? $woocommerce->cart->get_cart_contents_total()+WC()->cart->get_shipping_total() :$amount;
			$amount = $this->formatCurrencyAmount($amount, get_woocommerce_currency());

			
            if (WC()->session->get('destination') === true) {
				$transfer_data = array(
				'destination' => WC()->session->get('merchantId'),
				'amount' => $amount
				);
				$on_behalf_of = WC()->session->get('merchantId');
			}
			else {
				$transfer_data = null;
				$on_behalf_of = null;
			}
			try{
				$request = array(
					'customer' => $customerId,
					'source' =>$payment_source,
					'description' => sprintf('#%s, %s', $order_id, $order->get_billing_email()),				"capture_method" => "manual",
					'confirm' => false,
					'confirmation_method' => 'automatic',
					'amount' => $amount,
					'transfer_data'=>$transfer_data,
					'on_behalf_of' => $on_behalf_of,
					'currency' => get_woocommerce_currency(),
					'statement_descriptor_suffix' => $statement_descriptor
				);
				$log = new My_WC_LogTracking();
				$intent =\Stripe\PaymentIntent::create($request);
				$response = ["Response"=>$intent->charges, "Amount"=>$amount];
				$order_data = ["Customer Id"=>$customerId, "Email"=>$customerEmail];
				
				$intent_create = isset($intent->charges->data[0]->created) ? $intent->charges->data[0]->created : null;

				$intent_status = isset($intent->charges->data[0]->status) ? $intent->charges->data[0]->status : null;

				$log->payment_request(
					$order_data, "stripe",
					 date('r', $intent_create)."UTC",
					 $response,
					 $intent_status,
					 $request,
					 $intent);
				$checkout_url = wc_get_checkout_url();
				if( strpos( $checkout_url, '?') !== false ) {
					$checkout_url .= '&order='.$order->get_id();
				} else {
					$checkout_url .= '?order='.$order->get_id();
				}
				
		
			}catch(Exception $e) {
				wc_add_notice( $e->getMessage(), 'error' );
				/* translators: error message */
				$order->update_status( 'failed' );
				return array(
					'result'   => 'fail',
					'redirect' => '',
					'refresh' => true
				);
			}
				/* Intent Request */
              
		/* CAPTCHA START */
				$secret =isset($_SESSION['site_keys']['site_key_secret']) && $_SESSION['site_keys']['site_key_secret']!=null?$_SESSION['site_keys']['site_key_secret']:'NO-SECRET-KEY';
				if($secret && isset($_POST['g-recaptcha-response']) && $_POST['g-recaptcha-response']!=null):
					//your site secret key
					$verifyURL = 'https://www.google.com/recaptcha/api/siteverify?';
						$post_data = http_build_query(
							array(
								'secret' => $secret,
								'response' => $_POST['g-recaptcha-response'],
								'remoteip' => $_SERVER["REMOTE_ADDR"]
							)
						);
						$ch =  curl_init($verifyURL);
						curl_setopt($ch, CURLOPT_POST, 1);
						curl_setopt($ch, CURLOPT_POSTFIELDS, $post_data);
						curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
						curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
						curl_setopt($ch, CURLOPT_HTTPAUTH, CURLAUTH_BASIC);
						curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
						curl_setopt($ch, CURLOPT_TIMEOUT, 5);
						curl_setopt($ch, CURLOPT_HTTPHEADER, array('Accept: application/json', 'Content-type: application/x-www-form-urlencoded'));
						$verifyResponse = curl_exec($ch);
						curl_close($ch);
					//get verify response data
					$responseData = json_decode($verifyResponse);
					$log->captcha_data($verifyResponse,$_POST['g-recaptcha-response'],$responseData->success);
					//$responseresult = json_decode(json_encode($responseData),true);
					if(!$responseData->success):			
							$message = "Please reload your page  and try again. If the issue still persists,please contact 'enterprise@quivers.com' for further assistance";
							//wc_add_notice($message , 'error' );	
							// return array(
							// 	'result'   => 'fail',
							// 	'redirect' => '',
							// 	'refresh' => true
							// );
					endif;	
				endif;
				/* CAPTCHA START */
				/* INTENT CONFIRM */
				$intent->confirm([
					'return_url' => $checkout_url
					]);
	
				if ($intent["status"]== "requires_action"|| $intent["status"]== "requires_source_action"){
					WC()->session->set('intent_id',$intent->id);
					WC()->session->set('order_complete_url',$this->get_return_url( $order ));
					$intent = \Stripe\PaymentIntent::retrieve(WC()->session->get('intent_id'));
					return array(
						'result' => 'success',
						'redirect' => $intent->next_action->redirect_to_url->url
					);
			
				}
				else if ($intent["status"]== "requires_capture"){
						$order->payment_complete($intent->charges->data[0]->id);
						wc_reduce_stock_levels($order->get_id());
						$log->payment_complete(json_encode($intent));
						// Redirect to the thank you page
						return array(
							'result' => 'success',
							'redirect' => $this->get_return_url( $order )
						);
				}
				/* INTENT CONFIRM END*/
         } catch(\Stripe\Error\ApiConnection $e) {
				// Network problem, perhaps try again.
				wc_add_notice( 'Unable to connect to payment gateway server. Please try again!', 'error' );
				/* translators: error message */
				$order->update_status( 'failed' );

				return array(
					'result'   => 'fail',
					'redirect' => '',
					'refresh' => true
				);
			} catch(\Stripe\Error\InvalidRequest $e) {
				// You screwed up in your programming. Shouldn't happen!
				wc_add_notice( 'Invalid Request', 'error' );
				/* translators: error message */
				$order->update_status( 'failed' );

				return array(
					'result'   => 'fail',
					'redirect' => '',
					'refresh' => true
				);
			} catch(\Stripe\Error\Api $e) {
				// Stripe's servers are down!
				wc_add_notice( 'Unable to connect to payment gateway server. Please try again!', 'error' );
				/* translators: error message */
				$order->update_status( 'failed' );

				return array(
					'result'   => 'fail',
					'redirect' => '',
					'refresh' => true
				);
			} catch(\Stripe\Error\Card $e) {
				wc_add_notice( $e->getMessage().' Please verify the details before trying again. If the issue persists please contact Quivers at 805-250-4555', 'error' );
				/* translators: error message */
				$order->update_status( 'failed' );

				return array(
					'result'   => 'fail',
					'redirect' => '',
					'refresh' => true
				);
			}
			catch(Exception $e) {
				wc_add_notice( $e->getMessage(), 'error' );
				/* translators: error message */
				$order->update_status( 'failed' );

				return array(
					'result'   => 'fail',
					'redirect' => '',
					'refresh' => true
				);
			}
    	        $order->payment_complete($intent->charges->data[0]->id);
    	        $order->reduce_order_stock();
	
	 	}
	
		public	function formatCurrencyAmount($amount, $currency)
 		{
 	        // Stripe takes amounts as the smallest currency unit. ...
 	        if (empty($amount)) {
 	            return null;
 	        }
 	        $multiple = 100;
 	        if (in_array(
 	            $currency,
 	            array('BIF', 'DJF', 'JPY', 'KRW', 'PYG', 'VND', 'XAF', 'XPF', 'CLP', 'GNF', 'KMF', 'MGA', 'RWF', 'VUV', 'XOF'),
 	            true
 	        )) {
 	            $multiple = 1;
 	        }

 	        return sprintf('%01.0f', (float)$amount * $multiple);
 	    }
	}
}
}