<?php
/**
 * Plugin Name: Quivers PayPal
 * Description: Used to invoice, override create refunds api, use taxation api, save basic settings.
 * Version: 1.4.1
 * Author: Quivers Inc.
 * Author URI: nitesh.yadav@quivers.com
 * License: GPL2
 */
use PayPal\Rest\ApiContext;
use PayPal\Auth\OAuthTokenCredential;
use PayPal\Api\Amount;
use PayPal\Api\Details;
use PayPal\Api\ExecutePayment;
use PayPal\Api\Item;
use PayPal\Api\ItemList;
use PayPal\Api\Payer;
use PayPal\Api\Payment;
use PayPal\Api\PaymentExecution;
use PayPal\Api\RedirectUrls;
use PayPal\Api\Transaction;
use PayPal\Exception\PayPalConnectionException;


// define constant - path to paypal plugin file
define( 'WOO_PAYMENT_DIR', plugin_dir_path( __FILE__ )); 

include_once( ABSPATH . 'wp-admin/includes/plugin.php' );
if ( is_plugin_active( 'woocommerce/woocommerce.php' ) ){
// hooks the function woo_payment_gatewat to "plugins_loaded" action
add_action( 'plugins_loaded', 'quivers_payment_gateway' );

function quivers_payment_gateway() {
    class Quivers_PayPal_Gateway extends WC_Payment_Gateway {
        /**
         * API Context used for PayPal Authorization
         * @var null
         */
        public $apiContext = null;
        
        public function __construct() {
            $this->id = 'quivers_paypal'; 
            $this->method_title = __( 'Quivers-Pay PayPal', 'woodev_payment' );  
            $this->method_description = __( 'WooCommerce Payment Gateway for Quivers', 'quivers_paypal' );
            $this->has_fields = false;
            $this->supports = array( 'products' );
            $this->get_paypal_sdk();
            $this->init_form_fields();
            $this->init_settings();
            // load the title after form fields are initialized
            $this->title = $this->get_option('title');
            $this->enabled = $this->get_option('enabled');
        
            add_action( 'check_woopaypal', array( $this, 'check_response') );
            // Save settings
            if ( is_admin() ) {
                // Versions over 2.0
                // Save our administration options. Since we are not going to be doing anything special
                // we have not defined 'process_admin_options' in this class so the method in the parent
                // class will be used instead
                add_action( 'woocommerce_update_options_payment_gateways_' . $this->id, array( $this, 'process_admin_options' ) );
            }
        }

        private function get_paypal_sdk() {
            require_once WOO_PAYMENT_DIR . 'lib/paypal-sdk/autoload.php';
        }

        public function init_form_fields() {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => __( 'Enable', 'quivers_paypal' ),
                    'type' => 'checkbox',
                    'label' => __( 'Quivers PayPal', 'quivers_paypal' ),
                    'default' => 'yes'
                ),
                'title' => array(
                    'title' => __( 'Title', 'quivers_paypal' ),
                    'type' => 'text',
                    'default' => 'Paypal'
                ),
                'client_id' => array(
                    'title' => __( 'Client ID', 'quivers_paypal' ),
                    'type' => 'text',
                    'default' => ''
                ),
                'client_secret' => array(
                    'title' => __( 'Client Secret', 'quivers_paypal' ),
                    'type' => 'password',
                    'default' => ''
                ),
                'paypal_sandbox_enabled' => array(
                    'title' => __( 'PayPal Sandbox Enabled', 'quivers_paypal' ),
                    'type' => 'select',
                    'default' => 'Yes',
                    'options' => array('Yes', 'No')
                ),
            );
        }

        private function get_api_context(){
            $client_id =  $this->get_option('client_id');
            $client_secret =  $this->get_option('client_secret');
            $this->apiContext = new ApiContext(new OAuthTokenCredential($client_id, $client_secret));
        }

        public function process_payment( $order_id ) {
            global $woocommerce;
            $order = new WC_Order( $order_id );
            $this->get_api_context();
            $mode = $this->get_option('paypal_sandbox_enabled');
            if($mode == 0){
                $this->apiContext->setConfig(array('mode' => 'sandbox', ));
            }
            else{
                $this->apiContext->setConfig(array('mode' => 'live', ));
            }
            $payer = new Payer();
            $payer->setPaymentMethod("paypal");

            $all_items = array();
            $subtotal = 0;
            // Products
            foreach ( $order->get_items( array( 'line_item', 'fee' ) ) as $item ) {
                $itemObject = new Item();
                $itemObject->setCurrency( get_woocommerce_currency() );
                if ( 'fee' === $item['type'] ) {
                    $itemObject->setName( __( 'Fee', 'woo_paypal' ) );
                    $itemObject->setQuantity(1);
                    $itemObject->setPrice( $item['line_total'] ); 
                    $subtotal += $item['line_total'];
                } else {
                    $product = $order->get_product_from_item( $item );
                    $sku = $product ? $product->get_sku() : '';
                    $itemObject->setName( $item['name'] );
                    $itemObject->setQuantity( $item['qty'] );
                    $itemObject->setPrice( $order->get_item_subtotal( $item, false ) );
                    $subtotal += $order->get_item_subtotal( $item, false ) * $item['qty'];
                    if( $sku ) {
                        $itemObject->setSku( $sku );
                    }  
                }
                $all_items[] = $itemObject;
            }

            // adding a line item with negative discount
            $order_discount_amt = $order->get_total_discount();
            if ($order_discount_amt) {
                $itemObject = new Item();
                $itemObject->setCurrency( get_woocommerce_currency() );
                $itemObject->setName( 'Discount' );
                $itemObject->setQuantity( 1 );
                $itemObject->setPrice( -$order_discount_amt );
                $subtotal -= $order_discount_amt;
                $all_items[] = $itemObject;
            }
             
            $itemList = new ItemList();
            $itemList->setItems( $all_items );
            // ### Additional payment details
            // Use this optional field to set additional
            // payment information such as tax, shipping
            // charges etc.
            $details = new Details();
            $details->setShipping( $order->get_total_shipping() )
                ->setTax( $order->get_total_tax() )
                ->setSubtotal( $subtotal );
            $amount = new Amount();
            $amount->setCurrency( get_woocommerce_currency() )
                ->setTotal( $order->get_total() )
                ->setDetails($details);
            $transaction = new Transaction();
            $transaction->setAmount($amount)
                ->setItemList($itemList)
                ->setInvoiceNumber(uniqid());
            $baseUrl = $this->get_return_url( $order );
            if( strpos( $baseUrl, '?') !== false ) {
                $baseUrl .= '&';
            } else {
                $baseUrl .= '?';
            }
            $redirectUrls = new RedirectUrls();
            $redirectUrls->setReturnUrl( $baseUrl . 'order_status=true&order_id=' . $order_id )
                ->setCancelUrl( htmlspecialchars_decode($order->get_cancel_order_url()) . '&order_status=cancel' );
            $payment = new Payment();
            $payment->setIntent('authorize')
                ->setPayer($payer)
                ->setRedirectUrls($redirectUrls)
                ->setTransactions(array($transaction));
            try {
                $payment->create($this->apiContext);
                $approvalUrl = $payment->getApprovalLink();
                return array(
                    'result' => 'success',
                    'redirect' => $approvalUrl
                );
            } catch (Exception $ex) {
                wc_add_notice(  $ex->getMessage(), 'error' );
            }
            return array(
                'result' => 'failure',
                'redirect' => ''
            );
        }

        public function check_response() {
            global $woocommerce;
            $startTime = microtime(true);
            if( isset( $_GET['order_status'] ) ) {
                $order_status = $_GET['order_status'];
                $order_id = $_GET['order_id'];
                if( $order_id == 0 || $order_id == '' ) {
                    return;
                }
                $order = new WC_Order( $order_id );
                if( $order->has_status('completed') || $order->has_status('processing')) {
                    return;
                }
                if( $order_status == 'true' ) {
                    $transactionId = '';
                    $this->get_api_context();
                    $mode = $this->get_option('paypal_sandbox_enabled');
                    if($mode == 0){
                        $this->apiContext->setConfig(array('mode' => 'sandbox', ));
                    }
                    else{
                        $this->apiContext->setConfig(array('mode' => 'live', ));
                    }
                    $paymentId = $_GET['paymentId'];
                    $payment = Payment::get($paymentId, $this->apiContext);
                    $execution = new PaymentExecution();
                    $execution->setPayerId($_GET['PayerID']);
                    $transaction = new Transaction();
                    $amount = new Amount();
                    $details = new Details();
                    $subtotal = 0;
                    // Products
                    foreach ( $order->get_items( array( 'line_item', 'fee' ) ) as $item ) {
                        if ( 'fee' === $item['type'] ) {
                            $subtotal += $item['line_total'];
                        } else {
                            $subtotal += $order->get_item_subtotal( $item, false ) * $item['qty'];
                        }
                    }
                    $order_discount_amt = $order->get_total_discount();
                    if ($order_discount_amt) {
                        $subtotal -= $order_discount_amt;
                    }

                    $details->setShipping( $order->get_total_shipping() )
                     ->setTax( $order->get_total_tax() )
                     ->setSubtotal( $subtotal );

                    $amount = new Amount();
                    $amount->setCurrency( get_woocommerce_currency() )
                         ->setTotal( $order->get_total() )
                        ->setDetails($details);
                        $transaction->setAmount($amount);
                        $execution->addTransaction($transaction);
                    try {
                        $result = $payment->execute($execution, $this->apiContext);
                        $transactionId = $this->getAuthorizationId($result);
                    } catch(PayPalConnectionException $ex) {
                        wc_add_notice( 'Connection Failed: Unable to execute payment' , 'error' );
                        $order->update_status('failed', sprintf( __( '%s payment failed! Transaction ID: %d', 'woocommerce' ), $this->title, $paymentId ) . ' ' . $ex->getMessage() );
                        return;
                    } catch (Exception $ex) {
                        wc_add_notice(  $ex->getMessage(), 'error' );
                        $order->update_status('failed', sprintf( __( '%s payment failed! Transaction ID: %d', 'woocommerce' ), $this->title, $paymentId ) . ' ' . $ex->getMessage() );
                        return;
                    }
                    if($transactionId == ''){
                        wc_add_notice(  'Payment Processing Failed', 'error' );
                        $order->update_status('failed', sprintf( __( '%s payment failed! Transaction ID: %d', 'woocommerce' ), $this->title, $paymentId ) . ' ' . $ex->getMessage() );
                        return;   
                    }
                    // Payment complete
                    $order->payment_complete( $transactionId );
                    $order_data = ["Amount"=>$amount->getTotal($order->get_total())];
                    $log = new My_WC_LogTracking();
                    $log->payment_request(
                        $order_data,
                         "PayPal",
                          date('r', $startTime)."UTC",
                           ["Transaction Id"=>$transactionId], 
                           "success",null,null
                        );
                    // Add order note
                    $order->add_order_note( sprintf( __( '%s payment approved! Trnsaction ID: %s', 'woocommerce' ), $this->title, $transactionId ) );
                    // Remove cart
                    $woocommerce->cart->empty_cart();
                }
                // commented out this code, because if user cancels the order, he will be redirected to cancel order page and 
                // order gets cancelled with inbuilt cancel order functionality
                // if( $order_status == 'cancel' ) { 
                //     $order = new WC_Order( $order_id );
                //     $order->update_status('cancelled', sprintf( __( '%s payment cancelled!', 'woocommerce' ), $this->title ) );
                // }
            }
            return;
        }

        protected function getAuthorizationId($paypalPayment) {
            $transactions = $paypalPayment->getTransactions();
            foreach ($transactions as $transaction) {
                $relatedResources = $transaction->getRelatedResources();
                foreach ($relatedResources as $relatedResource) {
                    $authorization = $relatedResource->getAuthorization();
                    if($authorization) {
                        $authorizationId = $authorization->getId();
                        return $authorizationId;
                    }
                }
            }
            return false;
        }
    }
}

/**
 * Add Gateway class to all payment gateway methods
 */
function woo_add_gateway_class( $methods ) {
    $methods[] = 'Quivers_PayPal_Gateway';
    return $methods;
}

//hooks the gateway class to "woocommerce_payment_gateways" filter action
add_filter( 'woocommerce_payment_gateways', 'woo_add_gateway_class' );

add_action( 'init', 'check_for_woopaypal' );

function check_for_woopaypal() {
    if( isset($_GET['order_status'])) {
      // Start the gateways
        WC()->payment_gateways();
        do_action( 'check_woopaypal' );
    }
}
}